<?php
declare(strict_types=1);

function al_h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function al_now(): string { return gmdate('c'); }

function al_load_config(): array {
  $cfg_file = __DIR__ . '/../config.php';
  $sample = __DIR__ . '/../config.sample.php';
  if (!file_exists($cfg_file)) copy($sample, $cfg_file);
  $cfg = require $cfg_file;

  // Initialize default admin password hash if empty
  if (empty($cfg['admin']['password_hash'])) {
    $cfg['admin']['password_hash'] = password_hash('changeme123!', PASSWORD_DEFAULT);
    file_put_contents($cfg_file, "<?php
return " . var_export($cfg, true) . ";
");
  }
  return $cfg;
}

function al_load_state(array $cfg): array {
  $p = $cfg['data_path'];
  if (!file_exists($p)) return [];
  $raw = file_get_contents($p);
  $j = json_decode($raw ?: '{}', true);
  return is_array($j) ? $j : [];
}

function al_save_state(array $cfg, array $state): void {
  $p = $cfg['data_path'];
  $dir = dirname($p);
  if (!is_dir($dir)) @mkdir($dir, 0775, true);
  file_put_contents($p, json_encode($state, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
}

function al_http_json(string $url, array $payload, int $timeout=12): array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS => 3,
    CURLOPT_CONNECTTIMEOUT => $timeout,
    CURLOPT_TIMEOUT => $timeout,
    CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
    CURLOPT_USERAGENT => 'MyBTC-AutoLinker-Client/1.0',
  ]);
  $body = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  $data = json_decode($body ?: '{}', true);
  if (!is_array($data)) $data = ['raw'=>$body];

  return ['ok'=>($code>=200 && $code<300), 'status'=>$code, 'data'=>$data, 'error'=>$err];
}

function al_http_get(string $url, int $timeout=12): array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS => 3,
    CURLOPT_CONNECTTIMEOUT => $timeout,
    CURLOPT_TIMEOUT => $timeout,
    CURLOPT_USERAGENT => 'MyBTC-AutoLinker-Client/1.0',
  ]);
  $body = curl_exec($ch);
  $err = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return ['ok'=>($body!==false), 'status'=>$code, 'body'=>($body===false?'':$body), 'error'=>$err];
}

function al_normalize_url(string $url): string {
  $url = trim($url);
  if ($url === '') return '';
  if (!preg_match('~^https?://~i', $url)) $url = 'https://' . $url;
  $u = parse_url($url);
  if (!$u || empty($u['host'])) return '';
  $scheme = strtolower($u['scheme'] ?? 'https');
  $host = strtolower($u['host']);
  $path = $u['path'] ?? '/';
  if ($path === '') $path = '/';
  return $scheme . '://' . $host . $path;
}
